; PONG-Boot: The game of PONG (first Atari game, 1972) in 512 bytes!
; This version: written and conceived by Eric Auer. Details below.

; *** BACKGROUND ***
; Some historic info from telespiele.de: In 1958, Willy Higinbotham
; modified a scope to simulate paddles and ball. In 1962, Steve
; Russel programmed SPACEWAR (on an 8 mio usd computer...). This
; fascinates student Nolan Bushnell. The first home videogame by
; Ralph Baer is patented in 1966. Finally, in 1971, Nolan Bushnnell
; and Bill Pitts try to sell "computer space" and "galaxy game"
; respectively. In 1972, Magnavox tries to sell a home videogame
; based on an idea by Ralph Baer. In the same year, Bushnell tries
; to sell a video game machine idea to a pinball machine company.
; They are not interested, so he founds ATARI to sell PONG.

; *** RULES ***
; The big success of PONG is that it is very easy to play. You
; cannot be too drunk to at least believe you would get it. Just
; move the paddle to hit the ball. In my 512 byte version, you
; use the up/down cursor keys to move the paddle. The other paddle
; is played by the computer. If a player misses the ball, the other
; player gets a point. I also have an improved 1024 byte version,
; available on request. Use the ESC key or reboot to leave.

; *** LICENSE ***
; This game is OPEN SOURCE FREEWARE: You may use it and read the
; code. I even allow you to modify the code. But before you spread
; modified versions, please tell me about your version so that I
; can add it to my collection. There are so many clones of PONG out
; there that I have no problems with offering yet another one. And
; my clone is, of course, special: It fits a single boot sector.
; *** By Eric Auer 1993 and 2003. eric -at- coli.uni-sb.de :-). ***

; *** USAGE ***
; To compile as sector: nasm -o bootpong.bot bootpong.asm
; To compile for boot menu: nasm -DNODATA -o pong.bin bootpong.asm
; * result will have more features but not suitable for floppies
; * which contain other data as well. Only for menus/empty floppies!
; To compile as COM program: nasm -DCOM -o pong.com bootpong.asm
; To install: copy the boot sector to an unused floppy and boot
; from it. Or install it as an entry in your favourite boot menu.
; * If you have DOS / Windows / Linux data on the floppy, you may
; * have to adjust the DOSDATA  area - otherwise, you may lose
; * access to the other contents of the floppy after installing PONG!
; For the COM program, no special install is needed.
; Compile with -DSIMPLEFIELD to have a simple field and save space.

%define BALL bp		; register var.:  ball position (lo: x, hi: y)
%define SCORES si	; register variable: scores (lo: user, hi: PC)
%define DIR di		; register variable: ball movement (lo=x, ...)

%ifndef COM
	org 7c00h	; boot sectors are loaded to 0:7c00h
%ifndef NODATA
	jmp short afterData
	nop
DOSDATA:	db 'PONGBOOT'	; below is for 1.44 MB FAT12 floppies only!
	db 0,2,1,1,0
	db 2,0e0h, 0, 40h, 11, 0f0h
	dw 9, 18, 2, 0, 0
	dw 0, 0, 0
	db 29h
	dd 0c001f001h	; serial number :-)
	db "NO NAME    ", "FAT12   "
	; the next lines seem to be FreeDOS boot specific - not needed
	; db 60h, 0
	; db 14, 0, 1, 0, 0, 0, 13h, 0, 0, 0, 21h, 0, 0, 0
%endif
afterData:
	cli		; needed on OLD x86, as we set SS and SP.
	cld
	xor ax,ax	; boot sectors are started by jumping to 0:7c00h
	mov ss,ax	; (interrupts are blocked for 1 instruction now)
	mov sp,7c00h	; allocate some stack space
	push cs
	pop ds		; make DS equal to CS, as in com programs
%else
	org 100h	; in com programs, starting and loading point
			; are ????:100h, and SS, DS, ES are equal to CS.
			; SP is also set to a sane value already.
%endif

; ------------------------------

; init_keyboard:
	sti		; make sure that interrupts are enabled
	mov ax,0305h	; set typematic rate
	xor bx,bx	; maximum rate, minimum delay!
	int 16h		; for speedy play :-)

; init_screen:
	mov ax,0b800h   ; video segment (adjust if your screen is mono)
        mov es,ax
        xor ax,ax	; set 40x25 video mode (available "everywhere")
        int 10h		; also clears the screen
	; mov ax,500h	; select page 0
	; int 10h
; initial_values:
	; variables: user, comp (bytes for paddle position)
	; already set to 0c 0c 01 01 in RAM (see end of this file)
        mov BALL,0d13h		; ball starts at row 14, column 20
        xor SCORES,SCORES	; start with no points ...
	mov DIR,0101h		; ball is first flying down / right

; ------------------------------

GAME:			; the main game loop starts here

; speed_control:
	xor ax,ax
	int 1ah		; get time to CX:DX
	mov bx,dx	; this tick (1/18 sec per tick)
sloop:	; xor ax,ax	; AH is unchanged anyway
	int 1ah		; get time again
	cmp bx,dx
	jz sloop	; keep asking until this tick has elapsed

; paint_borders:
	xor dx,dx	; upper left point
	mov ax,'+='	; draw playfield borders
%ifndef SIMPLEFIELD
	test bl,8	; based on time, change pattern
	jz borderLoop
	xchg al,ah
%endif
borderLoop:		; loop over columns
	call SET_ONE	; fill line
        mov dh,18h	; same at lower row
	call SET_ONE	; fill again
	add dx,0e801h	; next column, upper row again (0e8h is -18h)
	xchg al,ah	; alternating pattern
	cmp dl,40
	jb borderLoop
%ifndef SIMPLEFIELD
; paint_middle_line:
	mov dx,0113h	; draw middle line
	mov ax,':;'
	test bl,8	; based on time, change pattern
	jz middleLoop
	xchg al,ah
middleLoop:
	call SET_ONE
	inc dl
	call SET_ONE
	add dx,0ffh	; one left and one down
	xchg al,ah	; alternating pattern
	cmp dh,18h
	jb middleLoop
%endif

; clear_paddle:
        xor dx,dx		; left column
        mov dh,[ds:user]	; row of user paddle
        call CLR_PAD		; remove paddle at old position
        mov dh,[ds:comp]	; row of computer paddle
        mov dl,39		; right column
        call CLR_PAD		; remove paddle at old position

; take_key:
        mov ah,1		; is there a key press waiting?
        int 16h
        jz DONT_MOVE_USER	; otherwise, only move computer paddle
        xor ax,ax		; if yes, fetch that key
        int 16h			; AH is the scancode, AL is the ASCII
        mov al,[ds:user]	; get user paddle position
        cmp ah,48h		; "up" key
        jz userUp
        cmp ah,50h		; "down" key
        jz userDown
        cmp ah,1		; "escape" key
        jnz DONT_MOVE_USER

; quit:
%ifndef COM
        int 18h		; leave by jumping to ROM BASIC
	int 19h		; ... of reboot, if that failed
%else
	push cs
	pop es		; reset ES to normal value
	mov ax,0305h	; set typematic rate
	mov bx,010ch	; "normal values", kind of
	int 16h
	mov ax,4c00h	; return "okay" errorlevel and exit
	int 21h		; leave program
%endif

userUp:
        dec al		; one row up
        cmp al,2        ; !
        jnb moveUserPos	; if out of range, fall through, down again
userDown:
        inc al		; one row down
        cmp al,22       ; !
        ja DONT_MOVE_USER	; if out of range, ignore movement
moveUserPos:
        mov [ds:user],al	; update user paddle position
DONT_MOVE_USER:

; move_computer:
        mov ax,BALL
	cmp al,15		; ball in far left playfield region?
	jb DONT_MOVE_COMP	; computer "cannot see that far" ;-)
				; We GIVE THE USER A CHANCE by this!
	test al,3
	jz DONT_MOVE_COMP	; computer "cannot move that fast"...
        mov al,[ds:comp]	; computer paddle position
        cmp al,ah		; make computer paddle follow ball row
        jb computerDown
        ja computerUp
	jmp short DONT_MOVE_COMP
computerDown:
        inc al		; down one row
        cmp al,22       ; !
        jna move_comp_pos	; accept, unless out of range
computerUp:
        dec al		; up one row
        cmp al,2        ; !
        jb DONT_MOVE_COMP
move_comp_pos:
        mov [ds:comp],al	; update paddle position
DONT_MOVE_COMP:

; show_new_paddle:
        xor dx,dx		; left column
        mov dh,[ds:user]	; user paddle row
        call SET_PAD		; draw paddle at new position
        mov dl,39		; right column
        mov dh,[ds:comp]	; computer paddle row
        call SET_PAD		; draw paddle at new position

; update_ball:
        call CLR_BAL		; remove ball at old position
        mov ax,BALL		; get ball position
	mov bx,DIR		; get ball direction
	add al,bl	; XDIR	; update ball column (checked later)
        mov BALL,ax		; store new ball position
	; ** mov al,[ds:btmp]
	; ** add al,[ds:yslp]
bounce:	; ** mov dl,al
	mov ax,BALL
	; ** cmp dl,8	; !!
	; ** jb bok3
	add ah,bh	; YDIR	; update ball row
; check_border:
	cmp ah,1
        jb wall			; hit a wall?
        cmp ah,23       ; !
        jna ballOk		; hit a wall? otherwise we are done.
wall:	neg bh		; YDIR	; invert vertical ball speed
	; ** sub dl,8
	jmp short bounce	; take first step in new direction

ballOk:	; ** sub dl,8
	mov BALL,ax	; store new ball position
	mov DIR,bx	; store new ball direction
	mov dx,bx	; store ball direction
	; ** bok3: mov [ds:btmp],bl

; paddle_check:	; did a player hit or miss the ball?
	xor cx,cx	; CX is 0 now: computer to be checked
        cmp al,1        ; X part of BALL: left / almost left?
        jnb right_end
; left_end:
        or dl,dl	; XDIR
        jns ALL_OK	; if ball not flying left, fine
        mov bh,[ds:user]	; user is to be checked
	inc cx		; CX is 1 now: computer to be checked
        jmp short CHKHIT
right_end:
	cmp al,38       ; X part of BALL: right / almost right?
        jna ALL_OK
        or dl,dl	; XDIR	; if ball is flying left, fine
        js ALL_OK
        mov bh,[ds:comp]	; computer is to be checked
			; CL is 0 now: computer to be checked
CHKHIT:
	neg dl		; XDIR	; X direction always changes here!
	mov DIR,dx	; update ball direction!
        mov bl,ah	; compare ball row AH to paddle row BH
        dec bh		; MAGIC ;-)
        sub bl,bh
        jc missBall
        cmp bl,2	; MAGIC ;-)
        jna ALL_OK	; hitBall - X direction already changed...

missBall:		; one player missed, the other scores.
        mov ax,SCORES	; AL=scru (user) AH=scrc (computer) score
	mov bl,99	; maximum displayable score
        jcxz cerr	; 0 = computer missed, 1 = user missed
uerr:	cmp ah,bl
	jae storeScores
	inc ah		; point for the computer
	jmp short storeScores
cerr:	cmp al,bl
	jae storeScores
	inc ax		; point for the user
storeScores:
	mov SCORES,ax	; store updated scores

	; move ball to middle col after miss (xdir already flipped)
	; This makes life harder for the one who did NOT miss ->
	; towards more equal chances
        ; *** Add random position / direction here? ***
        mov ax,BALL
	mov al,19h
	mov BALL,ax

ALL_OK:			; now display both scores!
	mov ax,SCORES	; first, show user score
	mov dx,0209h	; 3rd row, 10th column: user score
	call printScore
	mov dl,29	; 3rd row, 30th column: computer score
	mov al,ah	; next, show computer score
	call printScore

%if 0
	mov ax,BALL
; throw_new_ball:	; if the ball has left the field, return it
; *** only needed if the ball can fly at non-unit speeds (...) ***
	test ax,8080h	; either X or Y negative?
	jnz newBall
	cmp al,39
	jg newBall
	cmp ah,24
	jg newBall
	jmp GAME		; main loop, next iteration
newBall:
	mov BALL,0d13h		; BALL starts at row 14, column 20
%endif

        call SET_BAL
	jmp GAME		; main loop, next iteration

; ------------------------------

printScore:		; AL is score, DX is position
	push ax		; store updated score!
	aam		; AH is 10s digit, AL is 1s digit now
	add ax,3030h	; turn into ASCII
	xchg al,ah
	call SET_ONE	; high digit first
	xchg al,ah
        inc dx		; next column
        call SET_ONE	; low digit next
	pop ax
        ret		; DL modified, AX preserved

; ------------------------------

SET_PAD:		; show pad on screen, position DX
        mov al,'|'	; or use 0dbh, filled block in IBM charset
        jmp short pad_xy
CLR_PAD:		; remove pad from screen, position DX
        mov al,' '
pad_xy: dec dh		; upper pad half size
        push cx
        mov cx,3	; total pad size
pd:     call SET_ONE
        inc dh		; next row
        loop pd
        pop cx
        ret

; ------------------------------

SET_BAL:		; show ball on screen
        mov al,'*'
        jmp short bal_xy
CLR_BAL:		; remove ball from screen
        mov al,' '
bal_xy: mov dx,BALL	; position is ball position
	call SET_ONE	; one char, default color
	ret

; ------------------------------

SET_ONE:		; write char AL at x / y DL / DH
        push di
	push ax
        mov di,ax	; backup char
        mov al,40
        mul dh		; Y
        add al,dl	; X
        adc ah,0
        add ax,ax	; 2 byte / char in video RAM
        xchg di,ax	; restore char / attrib, load pointer!
	mov ah,7	; default color
        stosw		; (write to ES:DI)
	pop ax
	pop di
        ret		; returns regs unchanged.

; ------------------------------

	; register variables:
; BP ("BALL") has ball coordinates
; SI ("SCORES") has user (lo) and computer (hi) scores
; DI ("DIR") has X (lo) and Y (hi) ball speed vector / direction
	; variables: user, comp
user    db 12	; paddle row
comp    db 12	; paddle row

%ifndef COM
                times   0x01fe-$+$$ db 0	; fill with 00
	; mark as bootable by putting 55 aa at relative offset 1fe.
boot:	db 55h,0aah
%endif

